/*
  (c) 2001 Nathan Hjelm <hjelmn@unm.edu>

  v0.4.1

  Driver to allow communication to the rio under OS X
*/

#include "driver.h"

#include <unistd.h>
#include <IOKit/IOCFPlugIn.h>

#if defined(__MacOSX__)

/* this variable is needed by both usb_open and usb_claiminterface */
CFRunLoopSourceRef runLoopSource;

struct usbdevice *usb_open(UInt32 idVendor, UInt32 idProduct, UInt32 timeout){
  io_iterator_t deviceIterator;
  io_service_t usbDevice;
  mach_port_t masterPort;

  IOCFPlugInInterface **plugInInterface = NULL;
  IONotificationPortRef gNotifyPort;

  CFMutableDictionaryRef matchingDict;

  IOReturn result;
  SInt32 score;
  UInt16 vendor, product;

  struct usbdevice *dev;

  /* Create a master port for communication with IOKit */
  result = IOMasterPort(MACH_PORT_NULL, &masterPort);
  if (result || !masterPort)
    return NULL;

  /* set up the matching dictionary for class IOUSBDevice and it's subclasses */
  if((matchingDict = IOServiceMatching(kIOUSBDeviceClassName)) == NULL){
    mach_port_deallocate(mach_task_self(), masterPort);
    return NULL;
  }

  /* add the device to the matching dictionary */
  CFDictionarySetValue(matchingDict, CFSTR(kUSBVendorName),
		       CFNumberCreate(kCFAllocatorDefault,
				      kCFNumberSInt32Type, &idVendor));
  CFDictionarySetValue(matchingDict, CFSTR(kUSBProductName),
		       CFNumberCreate(kCFAllocatorDefault,
				      kCFNumberSInt32Type, &idProduct));
  
  gNotifyPort = IONotificationPortCreate(masterPort);
  runLoopSource = IONotificationPortGetRunLoopSource(gNotifyPort);
  CFRunLoopAddSource(CFRunLoopGetCurrent(), runLoopSource,
		     kCFRunLoopDefaultMode);

  matchingDict = (CFMutableDictionaryRef) CFRetain(matchingDict);
  matchingDict = (CFMutableDictionaryRef) CFRetain(matchingDict);
  matchingDict = (CFMutableDictionaryRef) CFRetain(matchingDict);

  result = IOServiceAddMatchingNotification(gNotifyPort, kIOFirstMatchNotification,
					    matchingDict, NULL, NULL,
					    &deviceIterator);

  dev = (struct usbdevice *)malloc(sizeof(struct usbdevice));

  /* find device */
  while (usbDevice = IOIteratorNext(deviceIterator)){
    /* Create an intermediate plug-in */
    result = IOCreatePlugInInterfaceForService(usbDevice, kIOUSBDeviceUserClientTypeID,
					       kIOCFPlugInInterfaceID, &plugInInterface,
					       &score);

    result = IOObjectRelease(usbDevice);
    if (result || !plugInInterface)
      continue;

    (*plugInInterface)->QueryInterface(plugInInterface, CFUUIDGetUUIDBytes(kIOUSBDeviceInterfaceID),
				       (LPVOID)&dev->device);

    if (!dev->device)
      continue;

    result = (*(dev->device))->GetDeviceVendor(dev->device, &vendor);
    result = (*(dev->device))->GetDeviceProduct(dev->device, &product);
    if ((vendor != idVendor) || (product != idProduct))
      continue;

    result = (*(dev->device))->USBDeviceOpen(dev->device);
    if (result)
      continue;

    /* right vendor and product and it opened, device is ready to go */
    return dev;
  }
  
  free(dev);
  return NULL;
}

void usb_close(struct usbdevice *dev){
  (*(dev->device))->USBDeviceClose(dev->device);
}

int usb_claiminterface(struct usbdevice *dev, int interface){
  io_iterator_t iterator;
  io_service_t  usbInterface;  

  IOReturn kernResult, result;
  IOUSBFindInterfaceRequest request;
  IOCFPlugInInterface **plugInInterface = NULL;

  SInt32 score;

  request.bInterfaceClass = kIOUSBFindInterfaceDontCare;
  request.bInterfaceSubClass = kIOUSBFindInterfaceDontCare;
  request.bInterfaceProtocol = kIOUSBFindInterfaceDontCare;
  request.bAlternateSetting = kIOUSBFindInterfaceDontCare;

  (*(dev->device))->CreateInterfaceIterator(dev->device, &request, &iterator);
  
  usbInterface = IOIteratorNext(iterator);

  kernResult = IOCreatePlugInInterfaceForService(usbInterface,
						 kIOUSBInterfaceUserClientTypeID,
						 kIOCFPlugInInterfaceID,
						 &plugInInterface, &score);

  //No longer need the usbInterface object after getting the plug-in
  kernResult = IOObjectRelease(usbInterface);
  if ((kernResult != kIOReturnSuccess) || !plugInInterface)
    return -1;
  
  //Now create the device interface for the interface
  result = (*plugInInterface)->QueryInterface(plugInInterface,
					      CFUUIDGetUUIDBytes(kIOUSBInterfaceInterfaceID),
					      (LPVOID) &dev->interface);
  //No longer need the intermediate plug-in
  (*plugInInterface)->Release(plugInInterface);

  if (result || !dev->interface)
    return -1;
  
  /* claim the interface */
  kernResult = (*(dev->interface))->USBInterfaceOpen(dev->interface);
  if (kernResult)
    return -1;

  kernResult = (*(dev->interface))->CreateInterfaceAsyncEventSource(dev->interface, &runLoopSource);
  if (kernResult)
    return -1;
  CFRunLoopAddSource(CFRunLoopGetCurrent(), runLoopSource, kCFRunLoopDefaultMode);

  /* interface is claimed and async IO is set up return 0 */
  return 0;
}

int usb_releaseinterface(struct usbdevice *dev, int interface){
  if (dev->interface)
    (*(dev->interface))->USBInterfaceClose(dev->interface);

  return 0;
}

int usb_control_msg(struct usbdevice *dev, UInt16 request, UInt16 value, UInt16 index,
		    UInt16 length, unsigned char *data){
  IOUSBDevRequest urequest;

  urequest.bmRequestType = USBmakebmRequestType(kUSBIn, kUSBVendor, kUSBDevice);
  urequest.bRequest = request;
  urequest.wValue = value;
  urequest.wIndex = index;
  urequest.wLength = length;
  urequest.pData = data;

  if((*(dev->device))->DeviceRequest(dev->device, &urequest) == kIOReturnSuccess)
    return 0;
  else
    return -1;
}

void compl(){
}

int usb_bulk_msg(struct usbdevice *dev, UInt8 direction, UInt16 ep, unsigned char *data, UInt32 size){
  IOReturn ret;

  switch(direction){
  case kUSBIn:
    ret = (*(dev->interface))->WritePipeAsync(dev->interface, ep, data, size, NULL, (void *)dev->interface);
    break;
  case kUSBOut:
    ret = (*(dev->interface))->ReadPipe(dev->interface, ep, data, &size);
    break;
  default:
    return -1;
  }

  if (ret == kIOReturnSuccess)
    return 0;
  else
    return -1;
}

int write_bulk(struct usbdevice *dev, unsigned char *data, u_int32_t size){
  return usb_bulk_msg(dev, kUSBIn, 0x2, data, size);
}

int read_bulk(struct usbdevice *dev, unsigned char *data, u_int32_t size){
  return usb_bulk_msg(dev, kUSBOut, 0x1, data, size);
}

int control_msg(struct usbdevice *dev, u_int8_t direction, u_int8_t request, u_int16_t value,
		u_int16_t index, u_int16_t length, unsigned char *buffer){
  return usb_control_msg(dev, request, value, index, length, buffer);
}

struct usbdevice *rio_usb_open(void){
  struct usbdevice *dev;
  if ((dev = usb_open(VENDOR_DIAMOND01, PRODUCT_RIO600, 5000)) != NULL)
    return dev;
  else if ((dev = usb_open(VENDOR_DIAMOND01, PRODUCT_RIO800, 5000)) != NULL)
    return dev;
  else if ((dev = usb_open(VENDOR_DIAMOND01, PRODUCT_PSAPLAY, 5000)) != NULL)
    return dev;
  else
    return NULL;
}
#endif /* __MacOSX__ */
