/*
  (c) 2001 Nathan Hjelm <hjelmn@unm.edu>

  Changes:
  v0.4
     - Macos X support added.
     - usbdevfs support added.

  TODO:
     - Code is very hacked up and sloppy, I need to clean it up.
     - Finish impementing error handling.

  This file contains the most of the functions that are listed in rio-cpp.h.
*/

#include "config.h"

#include <string>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>

#if defined (HAVE_LIBGEN_H)
#include <libgen.h>
#endif

#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>

#include <stdio.h>

#include "rio-cpp.h"
#include "rio_usb.h"
#include "cksum.h"
#include "hexdump.h"
#include "usbdevfs.h"

#define RIOERROR(x) { error = x; \
return error; }
#define RIODONE RIOERROR(0)

/* moved this here since it is only internal to library */
#ifdef linux
#define RIODEVICE  "/dev/usb/rio"
#elif defined(__FreeBSD__)
#define RIODEVICE "/dev/urio"
#elif defined(__NetBSD__)
#define RIODEVICE "/dev/urio"
#elif defined(__MacOSX__)
#define RIODEVICE ""
#endif

#include "driver.h"

/*
  rioInstance:
    Constructors that in arguments and create a rioInstance.
*/
rioInstance::rioInstance(const int rioNumber, ostream *outstream, ostream *dstream, int d){
  error = 0;
  progress = NULL;
  debug = d;
  errorO = dstream;
  output = outstream;
  thisptr = NULL;

  /* create the buffer that will be used with this instance */
  buffer = new u_int8_t[2048];

  fileName = new char[14];
  sprintf(fileName,"%s%i", RIODEVICE, rioNumber);

  if ((device = init()) == NULL){
    if (debug && errorO)
      *errorO<<"rioInstance: init failed"<<endl;
    delete fileName;
    fileName = NULL;
    error = -1;
  }else{
    buildFileList();

    resetFileIterator();

    fill_info();
    
    wake();
  }
}

/* close the rio */
rioInstance::~rioInstance(void){
  if (!device)
    return;

  /* send the command to close the mp3 player */
  send_command(RIO_POLLD);

  /* delete the file list */
  while(head){
    fileList *p = head->next;
    delete head;
    head = p;
  }

  /* delete buffer */
  delete[] buffer;

  usb_releaseinterface(device, 0);
  usb_close(device);
}

char *rioInstance::returnName(void) const{
  return name;
}

void rioInstance::setName(char *newname){
  int len = (strlen(newname) > 31) ? 32 : strlen(newname) + 1;

  delete[] name;
  name = new char[len];

  strncpy(name, newname, len);

  set_prefs();
}

int rioInstance::returnVolume(void) const{
  return volume;
}

void rioInstance::setVolume(int newvolume){
  wake();
  if (ison == false)
    return;

  /* make sure the volume is in range */
  if (volume > 20)
    volume = 20;
  else if (volume < 0)
    volume = 0;
  else
    volume = newvolume;

  set_prefs();
}

int rioInstance::returnUsage(void) const{
  return usedSpace;
}

int rioInstance::returnFree(void) const{
  return freeSpace;
}

double rioInstance::returnFirm(void) const{
  return firm;
}

long int rioInstance::returnTime(void) const{
  return totalTime;
}

int rioInstance::returnNumMemdev(void) const{
  return numMem;
}

int rioInstance::returnEquilizerState(void) const{
  return equilizerState;
}

int rioInstance::returnLightState(void) const{
  return lightState;
}

int rioInstance::returnContrast(void) const{
  return contrast;
}

int rioInstance::returnRepeatState(void) const{
  return repeatState;
}

int rioInstance::returnPlaylist(void) const{
  return playList;
}

bool rioInstance::exists(void) const{
  if (device)
    return true;
  else
    return false;
}

int rioInstance::returnType(void) const{
  return type;
}

void rioInstance::fill_info(void){
  wake();

  if (ison == false)
    return;

  if (send_command(RIO_PREFR) != 0){
    error = ERIORDY;
    return;
  }
  
  if ((int)buffer[0])
    read_block(RIO_MTS);
  else
    return;

  volume = buffer[0x0b];
  lightState = buffer[0x09];
  contrast = buffer[0xa] - 1;
  repeatState = buffer[0x07] % 4;
  playList = buffer[0x0f];

  equilizerState = buffer[0x04];

  name = new char[32];
  strncpy((char *)name, (char *)&buffer[0x040], 32);

  if (send_command(RIO_DESCP) != 0){
    error = ERIORDY;
    return;
  }

  if ((int)buffer[0])
    read_block(256);
  else
    return;

  if (strstr((char *)&buffer[0x080], "600") != NULL)
    type = RIO600;
  else if (strstr((char *)&buffer[0x080], "800") != NULL)
    type = RIO800;
  else
    type = PSAPLAY;

  firm = buffer[5] + (0.1) * (buffer[4] >> 4) + (0.01) * (buffer[4] & 0xf);
}

void rioInstance::update_info(void){
  wake();

  if (ison == false)
    return ;

  if (send_command(RIO_PREFR) != 0){
    error = ERIORDY;
    return;
  }
  
  if ((int)buffer[0])
    read_block(RIO_MTS);
  else
    return;

  volume = buffer[0x0b];
  lightState = buffer[0x09];
  contrast = buffer[0xa] - 1;
  repeatState = buffer[0x07] % 4;
  playList = buffer[0x0f];

  equilizerState = buffer[0x04];

  delete[] name;
  name = new char[32];
  strncpy((char *)name, (char *)&buffer[0x040], 32);

  if (send_command(RIO_DESCP) != 0){
    error = ERIORDY;
    return;
  }

  if ((int)buffer[0])
    read_block(256);
  else
    return;

  firm = buffer[5] + (0.1) * (buffer[4] >> 4) + (0.01) * (buffer[4] & 0xf);
}

/*
  rioInstance::upload:
     Function takes in a file name and optionally an artist and
  or a title and attemts to upload the file to the Rio.
  Returns 0 on success.
*/
int rioInstance::upload(const char *fileName, const char *artist, const char *title, const char *album){
  wake();

  if (ison == false)
    return ERIOOFF;

  Info info;

  // TODO: make it easier to support new file types
  if ((strstr(fileName, ".mp3") !=  NULL) || (strstr(fileName, ".MP3") !=  NULL))
    // MPEG Layer 3
    info = MP3getInfo(fileName);
  else if (strstr(fileName, ".wma") != NULL)
    // Winblowz Media
    info = WMAgetInfo(fileName);
  else
    // MISC File
    info = makeDownloadableInfo(fileName);

  if (info.data == NULL)
    RIOERROR(-1);

  if (info.data->size >= freeSpace)
    RIOERROR(ERIOFULL);

  if (title)
    strncpy((char *)info.data->title, title, 34);

  if (artist)
    strncpy((char *)info.data->artist, artist, 34);

  if (album)
    strncpy((char *)info.data->album, album, 34);

  if (start_upload() < 0)
    RIOERROR(-1025);

  if (output)
    *output<<basename((char *)fileName)<<": ";

  if (bulk_upload(info, fileName) < 0)
    RIOERROR(-1026);

  if (finish_upload(info) < 0)
    RIOERROR(-1027);

  RIODONE;
}

/*
  rioInstance::upload:
     Function takes in a list of files and attempts to
  upload them.
     
     If artist and album fields are non-null all mp3s get
  the string specified
*/
int rioInstance::upload(char **fileList, const char *artist, const char *album){
  char **fileName = fileList;
  Info info = {NULL, 0};
  ostream *tmp = output;

  /* i have chosen to suppress output from getInfo functions */
  output = NULL;

  wake();

  if (ison == false){
    output = tmp;
    RIOERROR(ERIOOFF);
  }

  for(;*fileName;fileName++){
    // TODO: make it easier to support new file types
    if ((strstr(*fileName, ".mp3") !=  NULL) || (strstr(*fileName, ".MP3") !=  NULL))
      // MPEG Layer 3
      info = MP3getInfo(*fileName);
    else if (strstr(*fileName, ".wma") != NULL)
      // Winblowz Media
      info = WMAgetInfo(*fileName);
    else
      // MISC File
      info = makeDownloadableInfo(*fileName);
    
    if (info.data == NULL)
      continue;
    
    if (info.data->size >= freeSpace)
      continue;
    
    if (album)
      strncpy((char *)info.data->album, album, 34);
    
    if (artist)
      strncpy((char *)info.data->artist, artist, 34);
    
    if (start_upload() < 0)
      continue;
    
    if (tmp){
      *tmp<<basename((char *)*fileName);
      if (strlen(*fileName) < 36)
	for (int i = 0 ; i < (36 - strlen(*fileName)) ; i++) tmp->put(' ');

      *tmp<<": ";
    }

    if (bulk_upload(info, *fileName) < 0)
      continue;
    
    if (finish_upload(info) < 0)
      continue;
  }

  output = tmp;
  delete info.data;

  RIODONE;
}

/*
  rioInstance::start_upload:
     Function tells the Rio that we are uploading.
*/
int rioInstance::start_upload(void){
  if (send_command(RIO_WRITE) != 0)
    RIOERROR(ERIORDY);

  /* should start with SRIORDY. */
  read_block();
  if (memcmp(buffer, "SRIORDY", 7) != 0)
    RIOERROR(-257);

  read_block();
  if (memcmp(buffer, "SRIODATA", 8) != 0){
    /* finally figured this out */
    if (memcmp(buffer, "SRIOFILE", 8) != 0){
      abort_transfer();
      
      read_block();

      start_upload();
    }else
      RIOERROR(-258);
  }
  
  RIODONE;
}

/*
  rioInstance::bulk_upload:
     Function takes in an Info structure and a file name
  and writes the file to the Rio. This should be called
  following start_upload.
*/
int rioInstance::bulk_upload(Info info, const char *fileName){
  int addfd, leftover, x, X;
  u_int8_t *fileBuffer;

  fileBuffer = new u_int8_t[RIO_FTS];

  if ((addfd = open(fileName, O_RDONLY, 0)) < 0)
    RIOERROR(-1);

  lseek(addfd, info.junk, SEEK_SET);
  X = (info.data->size / RIO_FTS);
  for (x = 0 ; x < X; x++){
    /* read in a chunk of file */
    read(addfd, fileBuffer, RIO_FTS);

    if (progress)
      progress(x, X, thisptr);
    
    if (write_block(fileBuffer, RIO_FTS) == -1)
      RIOERROR(-2);
  }

  /* last segment */ 
  leftover = info.data->size - (RIO_FTS * x);

  read(addfd, fileBuffer, leftover);

  if (write_block(fileBuffer, RIO_FTS) == -1)
    RIOERROR(-3);

  if (progress)
    progress(1, 1, thisptr);
  
  delete fileBuffer;

  RIODONE;
}

/*
  rioInstance::finish_upload:
     Function takes in an Info structure and
  writes the data field to the Rio. This should
  be called following bulk_upload.
*/
int rioInstance::finish_upload(Info info){
  int ret, crc;
  unsigned int *intp;

  info.data->file_no = numFile + 1;

#if BYTE_ORDER == BIG_ENDIAN
  rio_file *tmp = info.data;
  info.data = swap_file(info.data);
  delete tmp;
#endif

  crc = rio_crc32((unsigned char *)info.data, RIO_MTS);

  intp = (unsigned int *)buffer;
  intp[2] = crc; /* checksum */
  bcopy("CRIOINFO", buffer, 8);	
  ret = write_bulk(device, buffer, 64);
  
  if (write_bulk(device, (unsigned char *)info.data, RIO_MTS) < 0)
    RIOERROR(-2);

  if (send_command(0x60, 0, 0) != 0)
    RIOERROR(ERIORDY);

  if (send_command(0x66, 0, 0) != 0)
    RIOERROR(ERIORDY);

  read_block();

  if (memcmp(buffer, "SRIODONE", 8) != 0)
    RIOERROR(-1);

  RIODONE;
}

/*
  rioInstance::download:
     Function takes in the number of the file
  and attemt to download it from the Rio.

  Note: This only works with the following files:
  - Recorded WAVE files on the Rio 800
  - preferences.bin file
  - bookmarks.bin file
  - non-music files uploaded by rioutil

  It seems that, due to the riaa, diamond has
  made it so that wma and mp3 files CAN NOT
  be downloaded. I am currently looking for
  a way around this.
*/
int rioInstance::download(int fileno, const char *fileName){
  rio_file *downfile = new rio_file;
  unsigned char *downBuf = new unsigned char[RIO_FTS];
  Info dfile;
  int downfd;
  int size;

  get_file_info(downfile, fileno);

#if BYTE_ORDER == BIG_ENDIAN
  size = bswap_32(downfile->size);
#else
  size = downfile->size;
#endif

  send_command(RIO_READF);

  if ((int)buffer[0])
    read_block();
  else
    RIOERROR(-1);

  /* Write the info page to the rio */
  write_block((unsigned char *)downfile, sizeof(rio_file), false);

  delete downfile;

  if (memcmp(buffer, "SRIONOFL", 8) == 0)
    RIOERROR(-1);

  if (!fileName)
    /* we create a file with the same name as the file on the rio */
    downfd = creat((char *)&downfile->name, S_IRUSR | S_IWUSR);
  else
    /* we create a user-specified file */
    downfd = creat((char *)fileName, S_IRUSR | S_IWUSR);

  int i=0;
  int blocks = size/0x1000;

  bzero(buffer, 64);
  memcpy(buffer, "CRIODATA", 8);
  write_bulk(device, buffer, 64);
  read_block();
  
  if (blocks)
    for (;i<blocks;i++){
      read_bulk(device, downBuf, 0x1000);
      
      if (progress)
	progress(i, blocks, thisptr);
      
      write(downfd, downBuf, 0x1000);

      /* the rio seems to expect checksum */
      int crc32 = rio_crc32(downBuf, 0x1000);
      int *intp = (int *)buffer;

      bzero(buffer, 64);
      intp[2] = crc32;
      memcpy(buffer, "CRIODATA", 8);
      write_bulk(device, buffer, 64);

      if ((i+1)%4 == 0){
	read_block();

	if (memcmp(buffer, "SRIODONE", 8) == 0){
	  if (progress)
	    progress(1, 1, thisptr);

	    delete[] downBuf;
	    close(downfd);
	    RIODONE;
	}
      }
    }

  int leftover = size - (i * 0x1000);

  if (leftover){
    read_bulk(device, downBuf, 0x1000);
    write(downfd, downBuf, leftover);
    
    int crc32 = rio_crc32(downBuf, 0x1000);
    int *intp = (int *)buffer;
    
    bzero(buffer, 64);
    intp[2] = crc32;
    memcpy(buffer, "CRIODATA", 8);
    write_bulk(device, buffer, 64);
    i++;
  }

  if (progress)
    progress(1, 1, thisptr);

  close(downfd);
  
  /* 
     finish up a 16384 byte block (just zeros)
     dont know why this is needed, but i think it is 
  */
  for (;i%4;i++){
    read_bulk(device, downBuf, 0x1000);

    int crc32 = rio_crc32(downBuf, 0x1000);
    int *intp = (int *)buffer;
    
    bzero(buffer, 64);
    intp[2] = crc32;
    memcpy(buffer, "CRIODATA", 8);
    write_bulk(device, buffer, 64);
  }

  delete[] downBuf;

  read_block();

  if (memcmp(buffer, "SRIODONE", 8) != 0)
    RIOERROR(-1)
  else
    RIODONE;
}

/*
  rioInstance::abort_transfer:
     Function tell the Rio to abort the current transfer.
*/
int rioInstance::abort_transfer(void){
  bcopy("RIOABRT", buffer, 8);
  if ( this->exists() )
    return write_bulk(device, buffer, 64);
  else
    return -1;
}

/*
  rioInstance::init:
     Function opens the device file, sets the
  time and date on the rio, and returns the file
  descriptor.
*/
struct usbdevice *rioInstance::init(void)
{
  long int curr_time;
  struct timeval tv;
  struct timezone tz;
  struct tm *tmp;

  /*
   * the rio has no concept of timezones so we need to take
   * the local time into account when setting the time.
   * now using the (non)obselete gettimeofday function
   * i am not sure if this is the best way
   *
   */
  gettimeofday(&tv, &tz);
  tmp = localtime((const time_t *)&(tv.tv_sec));
  curr_time = tv.tv_sec - 60 * ( tz.tz_minuteswest - (60 * tmp->tm_isdst));

#if !defined(WITH_USBDEVFS) && !defined(__MacOSX__) && !defined(WITH_LIBUSB)
  device = new usbdevice;
  if ((device->fd = open(fileName, O_RDWR, 0666)) < 0){
    error = ENORIO;
    delete device;
    device = NULL;
    return device;
  }
#else
  if ((device = rio_usb_open()) == NULL){
    if (debug && errorO)
      *errorO<<"Could not open a Rio."<<endl;
    error = ENORIO;
    return NULL;
  }
#endif
  if (usb_claiminterface(device, 0)){
    if (debug && errorO)
      *errorO<<"Could not claim interface on device"<<endl;
    usb_close(device);
    return NULL;
  }

  if (send_command(RIO_TIMES, curr_time >> 16, curr_time & 0xffff) != 0){
    if (debug && errorO)
      *errorO<<"Set time command returned non-zero"<<endl;
    error = ERIORDY;
    usb_close(device);
    return NULL;
  }

  wake();
  
  return device;
}

int rioInstance::wake(){
  int *intp;

  if (!this->exists())
    RIOERROR(ENORIO);

  /*
    just in case the program was terminated
    improperly on last run
  */
  abort_transfer();

  if (send_command(0x65, 0, 0) != 0)
    RIOERROR(ERIORDY);

  if (send_command(0x60, 0, 0) != 0)
    RIOERROR(ERIORDY);
  
  intp = (int *)buffer;

  /* This is the value the rio seems to return if it is
     off or not ready. I am not sure if this is correct */
  if (intp[0] == 0x092f0041){
    ison = false;
    RIOERROR(ERIOOFF);
  }
  
  ison = true;

  RIODONE;
}

/*
  rioInstance::format:
     Function formats the memory of the instance.
*/
int rioInstance::format(void){
  if(send_command(RIO_FORMT, 0, 0) != 0)
    RIOERROR(ERIORDY);

  read_block(64);

  if (memcmp(buffer, "SRIOFMTD", 8) != 0)
    RIOERROR(-1);

  /* delete the file list */
  while(head){
    fileList *p = head->next;
    delete head;
    head = p;
  }

  RIODONE;
}

/*
  rioInstance::iterate:
     Function takes in a pointer to a function that
  takes a rio_file as its only argument and calls
  it will every file on the filelist.
*/
void rioInstance::iterate(void (*f)(rio_file *)){
  for (fileList *p = head ; p ; p = p->next)
    f(p->data);
}


/*
  rioInstance::resetFileIterator:
     Function sets current to head.
*/
void rioInstance::resetFileIterator(void){
  current = head;
}

/*
  I think there is a need for this type of iterator.
*/
rio_file *rioInstance::nextFile(void){
  if (!current)
    return NULL;

  rio_file *tmp = new rio_file;
  *tmp = *current->data;
  current=current->next;

  return tmp;
}

void rioInstance::setProgress(void (*f)(int x, int X, void *), void *pthis){
  progress = f;
  thisptr = pthis;
}

void rioInstance::setOutput(ostream &stream){
  output = &stream;
}

void rioInstance::setErrorOutput(ostream &stream){
  errorO = &stream;
}

void rioInstance::setDebugLevel(int i){
  debug = i;
}

/*
  rioInstance::read_block:
     Function takes in a block size and attempts to read
  it from the rio.
*/
int rioInstance::read_block(int block_size){
  int ret;

  ret = read_bulk(device, buffer, block_size);

  /* should start with command */
  if (ret < 0)
    RIOERROR(errno);
	
  if (ret != block_size)
    ret = block_size;

  if (debug && errorO)
    pretty_print_block(*errorO, buffer, block_size);

  return ret;
}

/*
  rioInstance::write_block:
     Function takes in a character buffer, buffer size, and
  a boolean.
     Writes the buffer to the Rio and checksums it if cksum is
  true.
*/
int rioInstance::write_block(unsigned char *writeBuffer, 
			     int bufferSize, bool cksum){
  int ret;
  static int error_count = 0;

  if (cksum){
    /* checksum it */
    unsigned int *intp = (unsigned int *)buffer;			
    intp[2] = rio_crc32(writeBuffer, bufferSize);
    
    /*  tell the rio the next block is data and write checksum */
    bcopy("CRIODATA", buffer, 8);
    ret = write_bulk(device, buffer, 64);
  }

  /* write the segment of the file */
  ret = write_bulk(device, writeBuffer, bufferSize);	

  if (debug > 2 && errorO)
    pretty_print_block(*errorO, writeBuffer, bufferSize);

  read_block();
  
  if ( cksum && (memcmp(buffer, "SRIODATA", 8) != 0) ){
    if (memcmp(buffer, "SRIOCRCF", 8) != 0)
      ret = -1;
    else{
      /* there is a crc error, count it and try again */
      if (error_count < 10){
	error_count++;
	
	write_block(writeBuffer, bufferSize);
      }else{
	/* error showed up too many times -- 10 is arbitrary */
	abort_transfer();
	ret = -1;
      }
    }
  }
  
  error_count = 0;
  return ret;
}

/*
  rioInstance::buildFileList:
     Function builds the file list for the instance and returns
  0 if completed successfully
*/
int rioInstance::buildFileList(void){
  /* we assume that head is not pointing anywhere */
  head = NULL;
  totalTime = 0;

  fileList **q = &head;

  int x = 0;
  rio_mem *memory = new rio_mem;

  if (send_command(0x66) != 0)
    RIOERROR(ERIORDY);

  if (send_command(RIO_POLLD) != 0)
    RIOERROR(ERIORDY);

  if (send_command(RIO_POLLD) != 0)
    RIOERROR(ERIORDY);

  wake();

  freeSpace = 0;
  usedSpace = 0;

  bzero(buffer, RIO_MTS);
  get_memdev_info(memory, 0);
  bzero(buffer, RIO_MTS);
  x++;

#if BYTE_ORDER == BIG_ENDIAN
    freeSpace += bswap_32(memory->free);
    usedSpace += bswap_32(memory->used);
#else
    freeSpace += memory->free;
    usedSpace += memory->used;
#endif

  numMem = x;
  
  delete memory;

  if (send_command(0x66) != 0)
    return ERIORDY;

  wake();

  x = 0;
  for (;;){
    rio_file *cur_file = new rio_file;
    bzero(buffer, RIO_MTS);
    if ((get_file_info(cur_file, x)) != 0){
      delete cur_file;
      break;
    }else{
      *q = new fileList(cur_file, NULL);
      q = &((*q)->next);
#if BYTE_ORDER == BIG_ENDIAN
      totalTime += bswap_32(cur_file->time);
#else
      totalTime += cur_file->time;
#endif
      x++;
    }
  }
  
  numFile = x;

  RIODONE;
}

/*
  rioInstance::set_prefs:
     Function writes the values that we have
  stored in this instance.
*/
int rioInstance::set_prefs(void){
  u_int8_t *prefbuf;

  if (send_command(RIO_PREFR) != 0)
    RIOERROR(ERIORDY);

  prefbuf = new u_int8_t[RIO_MTS];
  read_bulk(device, prefbuf, RIO_MTS);

  /* set name */
  strncpy((char *)(prefbuf + 0x40), name, 64);
  
  /* set volume */
  prefbuf[0x0b] = volume;

  if (send_command(RIO_PREFS) != 0){
    delete[] prefbuf;
    RIOERROR(ERIORDY);
  }

  read_block();
  write_bulk(device, prefbuf, 2048);
  read_block();

  delete[] prefbuf;
  RIODONE;
}

int rioInstance::refreshFileList(void){
  while(head){
    fileList *p = head->next;
    delete head;
    head = p;
  }

  buildFileList();

  RIODONE;
}

/*
  rioInstance::delete_file:
     Function takes in the number of the song deletes
  it and returns 0 if everything completes ok.
*/
int rioInstance::delete_file(int fileno){
  int ret;
  fileList *p;
  int i;

  if (fileno < 0)
    return -64;

  for (p = head, i = 0; i < fileno ; i++, p = p->next)
    if (!p)
      return -65;

  wake();

  if (send_command(RIO_DELET) != 0)
    RIOERROR(ERIORDY);
  
  read_block();
  
  if (memcmp(buffer, "SRIODELS", 8) != 0){
    error = -66;
    return error;
  }
  
  ret = write_bulk(device, (unsigned char *)p->data, RIO_MTS);
  
  if (ret < 0)
    return -1;
  
  read_block();
  
  if (memcmp(buffer, "SRIODELD", 8) != 0){
    error = -67;
    return error;
  }
  
  return 0;
}

/*
  rioInstance::get_file_info:
     Function takes in the number of the file
  and writes the info to argument 1.
*/
int rioInstance::get_file_info(rio_file *file, int fileno){
  int ret;
  
  if (send_command(RIO_FILEI, 0, fileno) != 0)
    return ERIORDY;

  if ((int)buffer[0])
    ret = read_block(RIO_MTS);
  else
    return -1;

  if (ret < 0)
    exit(1);
  
  if (int(buffer[0]))
    memcpy(file, buffer, sizeof(rio_file));
  else
    return -1;

  return 0;
}

/*
  rioInstance::get_memdev_info:
     Function takes in the memory device number (0 or 1)
  and writes the info to argument 1.
*/
int rioInstance::get_memdev_info(rio_mem *cur_mem, int dev){
  int ret;
	
  if (send_command(RIO_MEMRI, dev, 0) != 0)
    return ERIORDY;

  if ((int)buffer[0])
    ret = read_block(256);
  else
    return -129;

  if (ret < 0)
    exit(1);

  if ((int)buffer[0]){
    memcpy(cur_mem, buffer, sizeof(rio_mem));

#if BYTE_ORDER == BIG_ENDIAN
    cur_mem = swap_mem(cur_mem);
#endif
    
    return 0;
  }else
    return -130;
}

/*
  rioInstance::update:
     Function takes in a firmware file name (.lok)
  and updates the firmware on this instance with it.
  The function returns 0 is the update completes successfully.
*/
int rioInstance::update(const char *fileName){
  int firmfd;
  unsigned char *fileBuffer;
  struct stat statinfo;
  int size, x, *intp;
  int blocks, blocksize;

  fileBuffer = new unsigned char[RIO_FTS];

  if (stat(fileName, &statinfo) < 0)
    RIOERROR(errno);
  
  size = statinfo.st_size;

  /* try to open the firmware file */
  if ((firmfd = open(fileName, O_RDONLY)) <= 0)
    return errno;

  read(firmfd, buffer, 0x95);

  /* is this really a .lok file? */
  if ((type == RIO600) && (memcmp(buffer, "RIO600", 6) != 0))
    goto firmError;
  else if ((type == RIO800) && (memcmp(buffer, "RIO800", 6) != 0))
    goto firmError;
  else if ((type == PSAPLAY) && (memcmp(buffer, "NIKEPSA", 6) != 0)){
    goto firmError;
  }

  wake();

  /* four times seems to be what is needed *shrug* */
  if (send_command(0x6b, 1, 0) != 0)
    RIOERROR(ERIORDY);

  if (send_command(0x6b, 1, 0) != 0)
    RIOERROR(ERIORDY);

  if (send_command(0x6b, 1, 0) != 0)
    RIOERROR(ERIORDY);

  if (send_command(0x6b, 1, 0) != 0)
    RIOERROR(ERIORDY);

  read_block();
  
  /* the rio wants the size of the firmware file */
  bzero(buffer, 64);
  intp = (int *)buffer;

#if BYTE_ORDER == BIG_ENDIAN
  intp[0] = bswap_32(size);
#else
  intp[0] = size;
#endif

  write_block(buffer, 64, false);


  blocksize = 0x2000;
  blocks = size / blocksize;

  lseek(firmfd, 0, SEEK_SET);

  for (x = 0 ; x < blocks; x++){
    /* read in a chunk of file */
    read(firmfd, fileBuffer, blocksize);

    if (progress && x != 0)
      progress(x/2, blocks, thisptr);

    if (write_block(fileBuffer, blocksize, false) == -1)
      return -3;
  }

  lseek(firmfd, 0, SEEK_SET);

  for (x = 0 ; x < blocks; x++){
    /* read in a chunk of file */
    read(firmfd, fileBuffer, blocksize);
    
    if (progress && x != (blocks - 1))
      progress(blocks/2 + x/2, blocks, thisptr);

    if (write_block(fileBuffer, blocksize, false) < 0)
      return -3;

    /* the rio expects the first block to be sent three times */
    if (!x){
      write_block(fileBuffer, blocksize, false);
      write_block(fileBuffer, blocksize, false);
    }
  }

  delete[] fileBuffer;
  close(firmfd);

  RIODONE;

 firmError:
  delete[] fileBuffer;
  close(firmfd);

  RIOERROR(-2);
}

/*
  rioInstance::send_command:
     Function takes in a command, its arguments, and the
  command length and sends it to the Rio.
  Returns 0 if completed successfully.
*/
int rioInstance::send_command(int request, int value, int index, unsigned short length) const{
#if !defined(WITH_USBDEVFS) && !defined(__MacOSX__) && !defined(WITH_LIBUSB)
  struct RioCommand cmd;
  int ret = 0;

  cmd.timeout		= 50;
  cmd.requesttype	= 0; 		/* low level usb req. type. */
  cmd.request		= request; 	/* the actualy request      */
  cmd.value		= value;	/* value??                  */
  cmd.index		= index;	/* indexy thingy            */
  cmd.length		= length;	/* length?                  */
  cmd.buffer		= buffer;

  if ( ioctl (device->fd, RIO_RECV_COMMAND, &cmd) < 0)
    ret = errno;
  
  if (debug && errorO){
    errorO->form("cmd: len: 0x%04x rt: 0x%08x rq: 0x%08x va: 0x%08x id: 0x%08x", 
		cmd.length,
		cmd.requesttype,
		cmd.request,
		cmd.value,
		cmd.index
		);
    errorO->put('\n');
    
    pretty_print_block(*errorO, buffer, cmd.length);
  }

  return ret;
#else
  int ret =  control_msg(device, RIO_DIR_IN, request, value, index, length, buffer);
  
  if (debug && errorO){
    errorO->form("cmd: len: 0x%04x rt: 0x%08x rq: 0x%08x va: 0x%08x id: 0x%08x", 
		 length, 0, request, value, index);
    errorO->put('\n');
    
    pretty_print_block(*errorO, buffer, length);
  }

  return ret;
#endif
}
